/*
**                    --- simple.c ---
**
**  EXAMPLE CODE: Very simple terminal emulator.
**
**
*/

#include "windows.h"

#include "wsc.h"
#include "sioerror.h"
#include "ascii.h"
#include "simple.h"
#include "message.h"
#include "line.h"
#include "paint.h"
#include "about.h"

/* public globals */

HWND hMainWnd;            /* main window handle */
int OnLineFlag = FALSE;   /* TRUE: online */
int FatalFlag = FALSE;    /* TRUE: fatal error */

/* private globals */

static HMENU hMenu;
static HANDLE hInstance;
static int ThePort = COM1;
static int TheBaud = Baud19200;
static int WinWidth = 8 * NCOLS;
static int WinHeight = 12 * NROWS + 48;
static long BaudRateList[10]
       = {300L,600L,1200L,2400L,4800L,9600L,19200L,38400L,57600L,115200L};
static char Temp[1024];

/* miscellaneous functions */

int ErrorCheck(int);

#ifdef WIN32
int WINAPI
#else
int PASCAL
#endif
WinMain(HANDLE hInst,HANDLE hPrevInstance,LPSTR lpCmdLine, int nCmdShow)
{WNDCLASS  wc;
 MSG msg;
 BOOL Result;
 if(!hPrevInstance)
   {/* register main window class */
    wc.style = CS_HREDRAW | CS_VREDRAW;
    wc.lpfnWndProc = MainWndProc;
    wc.cbClsExtra = 0;
    wc.cbWndExtra = 0;
    wc.hInstance = hInst;
    wc.hIcon = LoadIcon(hInst, "SimpleIcon");
    wc.hCursor = LoadCursor(NULL, IDC_ARROW);
    wc.hbrBackground = GetStockObject(WHITE_BRUSH);
    wc.lpszMenuName =  "SimpleMenu";
    wc.lpszClassName = "SimpleWClass";
    Result = RegisterClass(&wc);
    if(!Result) return FALSE;
   }

 /* create main window */
 hInstance = hInst;
 hMainWnd = CreateWindow(
        "SimpleWClass",   "SIMPLE",       WS_OVERLAPPEDWINDOW,
        CW_USEDEFAULT,    CW_USEDEFAULT,
        WinWidth,         WinHeight,
        NULL,             NULL,
        hInstance,        NULL);
 ShowWindow(hMainWnd, nCmdShow);
 UpdateWindow(hMainWnd);
 hMenu = GetMenu(hMainWnd);

 /* window control loop */

 while(GetMessage(&msg,NULL,0,0))
   {
    TranslateMessage(&msg);
    DispatchMessage(&msg);
   }
 return (msg.wParam);
} /* end WinMain */

#ifdef WIN32
LRESULT CALLBACK
#else
long FAR PASCAL
#endif
MainWndProc(HWND hWindow,UINT message,WPARAM wParam,LPARAM lParam)
{
 UINT idTimer;
 HDC hDC;
 PAINTSTRUCT ps;
 int i;
 int TheChar;
 int Count;
 static FARPROC lpProcAbout;
 hMainWnd = hWindow;
 switch (message)
    {
     case WM_COMMAND:
         switch(wParam)
           {case MSG_ABOUT:
              DialogBox(hInstance,"AboutBox",hMainWnd,lpProcAbout);
              break;

            case MSG_DEBUG:
              break;

            case MSG_ONLINE:
              if(OnLineFlag) break;
              else
                {/* try to go on-line */
                 if(GoOnLine(ThePort,TheBaud,1024,128))
                   {/* we're online */
                    OnLineFlag = TRUE;
                    wsprintf((LPSTR)Temp,(LPSTR)"SIMPLE: COM%d online at %ld baud",
                       1+ThePort,BaudRateList[TheBaud]);
                    SetWindowText(hMainWnd,Temp);
                    CheckMenuItem(hMenu,MSG_OFFLINE,MF_BYCOMMAND | MF_UNCHECKED);
                    CheckMenuItem(hMenu,MSG_ONLINE,MF_BYCOMMAND | MF_CHECKED);
                    for(i=MSG_COM1;i<=MSG_COM4;i++)
                      EnableMenuItem(hMenu,i,MF_BYCOMMAND|MF_GRAYED);
                    for(i=MSG_300;i<=MSG_115200;i++)
                      EnableMenuItem(hMenu,i,MF_BYCOMMAND|MF_GRAYED);
                   }
                }
              break;

            case MSG_OFFLINE:
              if(!OnLineFlag) break;
              GoOffLine(ThePort);
              OnLineFlag = FALSE;
              SetWindowText(hMainWnd,"SIMPLE: Offline");
              CheckMenuItem(hMenu,MSG_ONLINE,MF_BYCOMMAND | MF_UNCHECKED);
              CheckMenuItem(hMenu,MSG_OFFLINE,MF_BYCOMMAND | MF_CHECKED);
              for(i=MSG_COM1;i<=MSG_COM4;i++)
                EnableMenuItem(hMenu,i,MF_BYCOMMAND|MF_ENABLED);
              for(i=MSG_300;i<=MSG_115200;i++)
                EnableMenuItem(hMenu,i,MF_BYCOMMAND|MF_ENABLED);
              break;

            case MSG_EXIT:
              DestroyWindow(hMainWnd);
              break;

            case MSG_COM1:
              CheckMenuItem(hMenu,MSG_COM1+ThePort,MF_BYCOMMAND | MF_UNCHECKED);
              ThePort = COM1;
              CheckMenuItem(hMenu,MSG_COM1,MF_BYCOMMAND | MF_CHECKED);
              break;

            case MSG_COM2:
              CheckMenuItem(hMenu,MSG_COM1+ThePort,MF_BYCOMMAND | MF_UNCHECKED);
              ThePort = COM2;
              CheckMenuItem(hMenu,MSG_COM2,MF_BYCOMMAND | MF_CHECKED);
              break;

            case MSG_COM3:
              CheckMenuItem(hMenu,MSG_COM1+ThePort,MF_BYCOMMAND | MF_UNCHECKED);
              ThePort = COM3;
              CheckMenuItem(hMenu,MSG_COM3,MF_BYCOMMAND | MF_CHECKED);
              break;

            case MSG_COM4:
              CheckMenuItem(hMenu,MSG_COM1+ThePort,MF_BYCOMMAND | MF_UNCHECKED);
              ThePort = COM4;
              CheckMenuItem(hMenu,MSG_COM4,MF_BYCOMMAND | MF_CHECKED);
              break;

            case MSG_300:
              CheckMenuItem(hMenu,MSG_300+TheBaud,MF_BYCOMMAND | MF_UNCHECKED);
              TheBaud = Baud300;
              CheckMenuItem(hMenu,MSG_300,MF_BYCOMMAND | MF_CHECKED);
              break;

            case MSG_1200:
              CheckMenuItem(hMenu,MSG_300+TheBaud,MF_BYCOMMAND | MF_UNCHECKED);
              TheBaud = Baud1200;
              CheckMenuItem(hMenu,MSG_1200,MF_BYCOMMAND | MF_CHECKED);
              break;

            case MSG_2400:
              CheckMenuItem(hMenu,MSG_300+TheBaud,MF_BYCOMMAND | MF_UNCHECKED);
              TheBaud = Baud2400;
              CheckMenuItem(hMenu,MSG_2400,MF_BYCOMMAND | MF_CHECKED);
              break;

            case MSG_4800:
              CheckMenuItem(hMenu,MSG_300+TheBaud,MF_BYCOMMAND | MF_UNCHECKED);
              TheBaud = Baud4800;
              CheckMenuItem(hMenu,MSG_4800,MF_BYCOMMAND | MF_CHECKED);
              break;

            case MSG_9600:
              CheckMenuItem(hMenu,MSG_300+TheBaud,MF_BYCOMMAND | MF_UNCHECKED);
              TheBaud = Baud9600;
              CheckMenuItem(hMenu,MSG_9600,MF_BYCOMMAND | MF_CHECKED);
              break;

            case MSG_19200:
              CheckMenuItem(hMenu,MSG_300+TheBaud,MF_BYCOMMAND | MF_UNCHECKED);
              TheBaud = Baud19200;
              CheckMenuItem(hMenu,MSG_19200,MF_BYCOMMAND | MF_CHECKED);
              break;

            case MSG_38400:
              CheckMenuItem(hMenu,MSG_300+TheBaud,MF_BYCOMMAND | MF_UNCHECKED);
              TheBaud = Baud38400;
              CheckMenuItem(hMenu,MSG_38400,MF_BYCOMMAND | MF_CHECKED);
              break;

            case MSG_57600:
              CheckMenuItem(hMenu,MSG_300+TheBaud,MF_BYCOMMAND | MF_UNCHECKED);
              TheBaud = Baud57600;
              CheckMenuItem(hMenu,MSG_57600,MF_BYCOMMAND | MF_CHECKED);
              break;

#ifdef WIN32
            case MSG_115200:
              CheckMenuItem(hMenu,MSG_300+TheBaud,MF_BYCOMMAND | MF_UNCHECKED);
              TheBaud = Baud115200;
              CheckMenuItem(hMenu,MSG_115200,MF_BYCOMMAND | MF_CHECKED);
              break;
#endif

            default:
              return (DefWindowProc(hMainWnd, message, wParam, lParam));
           }
         break;

    case WM_CREATE:

      /* check "OFFLINE" menu item */
      hMenu = GetMenu(hMainWnd);
      CheckMenuItem(hMenu,MSG_OFFLINE,MF_BYCOMMAND | MF_CHECKED);
      /* check default COM port */
      CheckMenuItem(hMenu,MSG_COM1+ThePort,MF_BYCOMMAND | MF_CHECKED);
      /* check default baud rate */
      CheckMenuItem(hMenu,MSG_300+TheBaud,MF_BYCOMMAND | MF_CHECKED);
      /* create AboutDlgProc() thunk */
      lpProcAbout = MakeProcInstance(AboutDlgProc, hInstance);
      /* initialize paint module */
      InitPaint();
      /* start timer */
      idTimer = SetTimer(hMainWnd,1,125,NULL);
      if(idTimer==0)
         {MessageBox(hMainWnd,"No timers remaining !","ERROR",MB_ICONEXCLAMATION | MB_OK);
          FatalFlag = TRUE;
         }
      break;

    case WM_CHAR:
      ErrorCheck(SioPutc(ThePort, (char)wParam ) );
      break;

    case WM_TIMER:
      /* fatal error ? */
      if(FatalFlag) break;
      if(!OnLineFlag) break;
      /* fetch line of up to 1024 chars */
      Count = 0;
      for(i=0;i<1024;i++)
        {TheChar = SioGetc(ThePort);
         /* character available ? */
         if(TheChar==WSC_NO_DATA) break;
         else if(ErrorCheck(TheChar)<0) break;
         Temp[Count++] = TheChar;
         if((char)TheChar==(char)LF) break;
        } /* end while */
      if(Count>0) WriteTheString(Temp,Count);
      break;

    case WM_SETFOCUS:
      /* create client area caret */
      CreateCaret(hMainWnd,NULL,3,10);
      SetCaretPos(GetXposition(),GetYposition());
      ShowCaret(hMainWnd);
      ShowCaret(hMainWnd);
      break;

    case WM_KILLFOCUS:
      DestroyCaret();
      break;

    case WM_PAINT:
      HideCaret(hMainWnd);
      hDC = BeginPaint(hMainWnd, &ps);
      SetMapMode(hDC,MM_ANISOTROPIC);
      SelectObject(hDC, GetStockObject(OEM_FIXED_FONT) );
      PaintMain(hDC,&ps);
      EndPaint(hMainWnd,&ps);
      SetCaretPos(GetXposition(),GetYposition());
      ShowCaret(hMainWnd);
      break;

    case WM_DESTROY:
      GoOffLine(ThePort);
      SetWindowText(hMainWnd,"SIMPLE: Offline");
      CheckMenuItem(hMenu,MSG_OFFLINE,MF_BYCOMMAND | MF_CHECKED);
      CheckMenuItem(hMenu,MSG_ONLINE,MF_BYCOMMAND | MF_UNCHECKED);
      if(idTimer) KillTimer(hMainWnd,idTimer);
      PostQuitMessage(0);
      break;

    default:
      return (DefWindowProc(hMainWnd, message, wParam, lParam));
   }
 return 0;

} /* end MainWndProc */

int ErrorCheck(int Code)
{/* trap PCL error codes */
 if(Code<0)
     {SioError(Code,"");
      SioRxClear(ThePort);
      SioDone(ThePort);
      FatalFlag = TRUE;
     }
 return Code;
} /* end ErrorCheck */
